/*************************************************************************************************
 * The MT-safe API of Hyper Estraier
 *                                                      Copyright (C) 2004-2005 Mikio Hirabayashi
 * This file is part of Hyper Estraier.
 * Hyper Estraier is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License or any later version.  Hyper Estraier is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with Hyper
 * Estraier; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA.
 *************************************************************************************************/


#ifndef _ESTMTDB_H                       /* duplication check */
#define _ESTMTDB_H

#if defined(__cplusplus)                 /* export for C++ */
extern "C" {
#endif


#include <estraier.h>
#include <cabin.h>
#include <pthread.h>
#include <stdlib.h>



/*************************************************************************************************
 * API for MT-safe database
 *************************************************************************************************/


typedef struct {                         /* type of structure for a MT-safe database */
  ESTDB *db;                             /* database connection */
  char *path;                            /* real path of the directory */
  pthread_mutex_t mutex;                 /* mutex for each connection */
} ESTMTDB;


/* Open a database. */
ESTMTDB *est_mtdb_open(const char *name, int omode, int *ecp);


/* Close a database. */
int est_mtdb_close(ESTMTDB *db, int *ecp);


/* Get the last happended error code of a database. */
int est_mtdb_error(ESTMTDB *db);


/* Check whether a database has a fatal error. */
int est_mtdb_fatal(ESTMTDB *db);


/* Flush index words in the cache of a database. */
int est_mtdb_flush(ESTMTDB *db, int max);


/* Synchronize updating contents of a database. */
int est_mtdb_sync(ESTMTDB *db);


/* Optimize a database. */
int est_mtdb_optimize(ESTMTDB *db, int options);


/* Add a document to a database. */
int est_mtdb_put_doc(ESTMTDB *db, ESTDOC *doc, int options);


/* Remove a document from a database. */
int est_mtdb_out_doc(ESTMTDB *db, int id, int options);


/* Retrieve a document in a database. */
ESTDOC *est_mtdb_get_doc(ESTMTDB *db, int id, int options);


/* Retrieve the value of an attribute of a document in a database. */
char *est_mtdb_get_doc_attr(ESTMTDB *db, int id, const char *name);


/* Get the ID of a document spacified by URI. */
int est_mtdb_uri_to_id(ESTMTDB *db, const char *uri);


/* Extract keywords of a document object. */
CBMAP *est_mtdb_etch_doc(ESTMTDB *db, ESTDOC *doc, int max);


/* Initialize the iterator of a database. */
int est_mtdb_iter_init(ESTMTDB *db);


/* Get the next ID of the iterator of a database. */
int est_mtdb_iter_next(ESTMTDB *db);


/* Get the name of a database. */
const char *est_mtdb_name(ESTMTDB *db);


/* Get the number of documents in a database. */
int est_mtdb_doc_num(ESTMTDB *db);


/* Get the number of unique words in a database. */
int est_mtdb_word_num(ESTMTDB *db);


/* Get the size of a database. */
double est_mtdb_size(ESTMTDB *db);


/* Search documents corresponding a condition for a database. */
int *est_mtdb_search(ESTMTDB *db, ESTCOND *cond, int *nump, CBMAP *hints);


/* Set the maximum size of the cache memory of a database. */
void est_mtdb_set_cache_size(ESTMTDB *db, size_t size, int anum, int tnum);


/* Set the special cache for narrowing and sorting with document attributes. */
void est_mtdb_set_special_cache(ESTMTDB *db, const char *name, int num);



/*************************************************************************************************
 * features for experts
 *************************************************************************************************/


/* Edit attributes of a document object in a database. */
int est_mtdb_edit_doc(ESTMTDB *db, ESTDOC *doc);


/* Add a piece of meta data to a database. */
void est_mtdb_add_meta(ESTMTDB *db, const char *name, const char *value);


/* Get a list of names of meta data of a database. */
CBLIST *est_mtdb_meta_names(ESTMTDB *db);


/* Get the value of a piece of meta data of a database. */
char *est_mtdb_meta(ESTMTDB *db, const char *name);


/* Get the number of records in the cache memory of a database. */
int est_mtdb_cache_num(ESTMTDB *db);

/* Set the callback function to inform of database events. */
void est_mtdb_set_informer(ESTMTDB *db, void (*func)(const char *));


/* Set the callback function to create a vector of keywords of a document. */
void est_mtdb_set_vectorizer(ESTMTDB *db, CBMAP *(*func)(void *, int, void *), void *data);


/* Fill the cache for keys for TF-IDF. */
void est_mtdb_fill_key_cache(ESTMTDB *db);



#if defined(__cplusplus)                 /* export for C++ */
}
#endif

#endif                                   /* duplication check */


/* END OF FILE */
